#include "videoplayer.h"

#include <QtWidgets>

#include <glib.h>
#include <gst/gst.h>
#include <gst/rtsp/gstrtsptransport.h>
#include <gst/video/videooverlay.h>


static void
pad_added_cb (GstElement *src, GstPad *new_pad, GstElement *peer)
{
    g_print ("Received new pad '%s' from '%s':\n", GST_PAD_NAME (new_pad), GST_ELEMENT_NAME (src));

    (void)src;        // remove unused variable warning
    GstPad *sinkpad = gst_element_get_static_pad (peer, "sink");
    GstCaps *new_pad_caps = nullptr;
    GstStructure *new_pad_struct = nullptr;
    const gchar *new_pad_type = nullptr;
    const GValue *new_pad_media = nullptr;

   /* Check the new pad's type */
    new_pad_caps = gst_pad_get_current_caps (new_pad);
    new_pad_struct = gst_caps_get_structure (new_pad_caps, 0);
    new_pad_type = gst_structure_get_name (new_pad_struct);
    if (g_str_has_prefix(new_pad_type, "video/x-h264")) {
        // h264 video using a filesrc pipeline
        GstPadLinkReturn ret = gst_pad_link (new_pad, sinkpad);
        if (GST_PAD_LINK_FAILED (ret)) {
            g_print ("Type is '%s' but link failed.\n", new_pad_type);
        } else {
            g_print ("Link succeeded (type '%s').\n", new_pad_type);
        }
        goto done;
    }
    if (g_str_has_prefix (new_pad_type, "application/x-rtp")) {
        new_pad_media = gst_structure_get_value (new_pad_struct, "media");
        // g_print (new_pad_media);

        if (g_str_has_prefix (g_value_get_string (new_pad_media), "video")) {
            /* Attempt the link */
            GstPadLinkReturn ret = gst_pad_link (new_pad, sinkpad);
            if (GST_PAD_LINK_FAILED (ret)) {
                g_print ("Type is '%s' but link failed.\n", new_pad_type);
            } else {
                g_print ("Link succeeded (type '%s').\n", new_pad_type);
            }
        } else {
            g_print ("New pad has type application/x-rtp, but media is %s, not video. Ignoring.\n", g_value_get_string(new_pad_media));
            goto done;
        }
    } else {
        g_print ("New pad has type '%s' which is not application/x-rtp. Ignoring.\n", new_pad_type);
    }

done:
    gst_object_unref (sinkpad);
}


void VideoPlayer::print_status_of_all()
{
  if (!pipeline) {
    return;
  }

  auto it = gst_bin_iterate_elements(GST_BIN(pipeline));
  GValue value = G_VALUE_INIT;
  for (GstIteratorResult r = gst_iterator_next(it, &value);
       r != GST_ITERATOR_DONE;
       r = gst_iterator_next(it, &value))
  {
    if (r == GST_ITERATOR_OK)
    {
      GstElement *e = static_cast<GstElement*>(g_value_peek_pointer(&value));
      GstState  current, pending;
      gst_element_get_state(e, &current, &pending, 100000);
      g_print("%s(%s), status = %s, pending = %s\n", G_VALUE_TYPE_NAME(&value), gst_element_get_name(e), gst_element_state_get_name(current), gst_element_state_get_name(pending));
    }
  }
}

VideoPlayer::VideoPlayer(QWidget *parent)
    : QWidget(parent)
{
    QAbstractButton *debugButton = new QPushButton(tr("?"));
    connect(debugButton, SIGNAL(clicked()), this, SLOT(debugSlot()));

    m_stopButton = new QPushButton(this);
    m_stopButton->setIcon(style()->standardIcon(QStyle::SP_MediaStop));
    m_stopButton->setEnabled(false);
    connect(m_stopButton, SIGNAL(clicked()), this, SLOT(stop()));

    m_playButton = new QPushButton(this);
    m_playButton->setIcon(style()->standardIcon(QStyle::SP_MediaPlay));
    connect(m_playButton, &QAbstractButton::clicked, this, &VideoPlayer::playPause);

    m_seekBackwardButton = new QPushButton("- 10s", this);
    connect (m_seekBackwardButton, SIGNAL(clicked()), this, SLOT(seekBackward10()));
    m_seekBackwardButton->setEnabled(false);

    m_seekForwardButton = new QPushButton("+ 10s", this);
    connect (m_seekForwardButton, SIGNAL(clicked()), this, SLOT(seekForward10()));
    m_seekForwardButton->setEnabled(false);

    m_nextFrameButton = new QPushButton(">|", this);
    connect (m_nextFrameButton, SIGNAL(clicked()), this, SLOT(nextFrame()));

    // -----------------------------------------------------------------
    // Options
    QCheckBox *tcpCheckBox = new QCheckBox("Use TCP protocol", this);
    tcpCheckBox->setChecked(use_tcp);
    connect(tcpCheckBox, SIGNAL(toggled(bool)), this, SLOT(useTcp(bool)));

    QCheckBox *rateControlCheckBox = new QCheckBox("Disable rate control", this);
    rateControlCheckBox->setChecked(!rate_control);
    connect(rateControlCheckBox, SIGNAL(toggled(bool)), this, SLOT(disableRateControl(bool)));

    QComboBox *frameModeBox = new QComboBox(this);
    QStringList frameModes = { "All", "Predicted", "Intra" };
    frameModeBox->addItems(frameModes);
    connect(frameModeBox, SIGNAL(activated(QString)), this, SLOT(setFrameMode(QString)));

    QSpinBox *frameIntervalBox = new QSpinBox(this);
    frameIntervalBox->setValue(1);
    frameIntervalBox->setEnabled(false);

    QLineEdit *rtspUrlEdit = new QLineEdit(this);
    connect (rtspUrlEdit, SIGNAL(textEdited(QString)), this, SLOT(setRtspUrl(QString)));
    rtspUrlEdit->setText(m_rtspUrls[0]);
    m_urlEdits.append(rtspUrlEdit);

    m_addVideoButton = new QPushButton("+");
    m_addVideoButton->setMaximumWidth(28);
    connect (m_addVideoButton, SIGNAL(clicked()), this, SLOT(addRtspUrlEdit()));

    QDoubleSpinBox *speedBox = new QDoubleSpinBox(this);
    speedBox->setRange(0.0, 32.0);
    speedBox->setSingleStep(0.1);
    speedBox->setValue(playSpeed); // defaults to 1.0
    connect(speedBox, SIGNAL(valueChanged(double)), this, SLOT(setPlaybackSpeed(double)));

    QCheckBox *reversePlaybackBox = new QCheckBox("Reverse playback", this);
    connect(reversePlaybackBox, SIGNAL(toggled(bool)), this, SLOT(setReversePlayback(bool)));


    m_urlsLayout = new QVBoxLayout();
    m_urlsLayout->addWidget(rtspUrlEdit);

    QFormLayout *optionsLayout = new QFormLayout();
    optionsLayout->addWidget(tcpCheckBox);
    optionsLayout->addWidget(rateControlCheckBox);
    optionsLayout->addRow("Frame mode", frameModeBox);
    optionsLayout->addRow("Intra frames interval", frameIntervalBox);
    optionsLayout->addRow("Rtsp URL", m_urlsLayout);
    optionsLayout->addRow("", m_addVideoButton);
    optionsLayout->addRow("Playback speed", speedBox);
    optionsLayout->addWidget(reversePlaybackBox);

    m_positionLabel = new QLabel("No position");
    m_positionLabel->setTextInteractionFlags(Qt::TextSelectableByMouse);
    m_positionLabel->setCursor(QCursor(Qt::IBeamCursor));
    QTimer *timer = new QTimer(this);
    connect(timer, SIGNAL(timeout()), this, SLOT(updatePosition()));
    // update position every 100ms
    timer->start(100);
    m_position = new QDateTimeEdit();
    m_position->setTimeSpec(Qt::UTC);
    m_position->setDisplayFormat("dd/MM/yyyy HH:mm:ss.zzz");
    //positionEdit->setCalendarPopup(true);
    connect(m_position, SIGNAL(editingFinished()), this, SLOT(seekToPosition()));

    // -----------------------------------------------------------------
    // Player controls
    QBoxLayout *controlLayout = new QHBoxLayout;
    controlLayout->setMargin(0);
    controlLayout->addWidget(debugButton);
    controlLayout->addWidget(m_seekBackwardButton);
    controlLayout->addWidget(m_stopButton);
    controlLayout->addWidget(m_playButton);
    controlLayout->addWidget(m_nextFrameButton);
    controlLayout->addWidget(m_seekForwardButton);
    controlLayout->addStretch();

    m_playersLayout = new QGridLayout();
    QWidget *firstPlayerWidget = new QWidget(this);
    m_videoDisplays.append(firstPlayerWidget);
    m_playersLayout->addWidget(firstPlayerWidget, 0, 0);
    firstPlayerWidget->setMinimumHeight(240);

    QBoxLayout *layout = new QVBoxLayout(this);
    layout->addLayout(optionsLayout);
    layout->addLayout(m_playersLayout, 1);
    layout->addWidget(m_positionLabel);
    layout->addWidget(m_position);
    layout->addLayout(controlLayout);

}

//void cb_message (GstBus *bus, GstMessage *msg, CustomData *data)
void cb_message (GstBus *bus, GstMessage *msg, GstElement *pipeline)
{
  (void)bus;          // remove unused variable warning
  GError *err;
  gchar *debug;
  int buffering_level = 0;

  switch (GST_MESSAGE_TYPE (msg)) {
  case GST_MESSAGE_ERROR:
    gst_message_parse_error (msg, &err, &debug);
    qCritical () << "Received Gstreamer error: " << debug;
    g_error_free (err);
    g_free (debug);
    gst_element_set_state (pipeline, GST_STATE_READY);
    break;
  case GST_MESSAGE_WARNING:
    gst_message_parse_warning (msg, &err, &debug);
    qWarning () << "Received Gstreamer warning: " << debug;
    g_error_free (err);
    g_free (debug);
    break;
  case GST_MESSAGE_BUFFERING:
    gst_message_parse_buffering (msg, &buffering_level);
    qDebug() << "Buffering level: " << buffering_level;
    /* Wait until buffering is complete before start/resume playing */
    if (buffering_level < 100) {
      gst_element_set_state (pipeline, GST_STATE_PAUSED);
    } else {
      gst_element_set_state (pipeline, GST_STATE_PLAYING);
    }
    break;
  default:
    /* Unhandled message */
    break;
  }
  return;
}

VideoPlayer::~VideoPlayer()
{
  gst_element_set_state (pipeline, GST_STATE_NULL);
  gst_object_unref (pipeline);
}

void VideoPlayer::initGst () {
  // prepare the pipeline
  gst_init (nullptr, nullptr);
  pipeline = gst_pipeline_new ("plop");

  GstBus *bus = gst_element_get_bus (pipeline);
  gst_bus_add_signal_watch (bus);
  g_signal_connect (bus, "message", G_CALLBACK (cb_message), pipeline);
}

void VideoPlayer::setGstTestVideo () {
  setGstTestVideo (0);
}

void VideoPlayer::setGstTestVideo (int pattern)
{
  initGst();

  gst_element_set_state (pipeline, GST_STATE_PAUSED);

  GstElement *src = gst_element_factory_make ("videotestsrc", nullptr);
  GstElement *time = gst_element_factory_make ("timeoverlay", nullptr);
  GstElement *sink = gst_element_factory_make ("glimagesink", nullptr);
  m_video_sinks.insert(0, sink);

  gst_bin_add_many (GST_BIN (pipeline), src, time, sink, nullptr);
  gst_element_link_many (src, time, sink, nullptr);
  g_object_set (src, "pattern", pattern, nullptr);
  g_object_set (time, "font-desc", "Sans, 24", nullptr);

  WId xwinid = m_videoDisplays[0]->winId();
  gst_video_overlay_set_window_handle (GST_VIDEO_OVERLAY (sink), xwinid);

  // Need to set newly added elements in the same state as the pipeline
  gst_element_set_state (pipeline, GST_STATE_PAUSED);

  return;
}


void VideoPlayer::setGstFileVideo ()
{
  qDebug() << "[setGstFileVideo] Entering function";
  initGst();

  GstElement *src = gst_element_factory_make ("filesrc", nullptr);
  GstElement *demux = gst_element_factory_make ("qtdemux", nullptr);
  GstElement *vdec = gst_element_factory_make ("avdec_h264", nullptr);
  GstElement *vqueue = gst_element_factory_make ("queue2", nullptr);
  GstElement *vconv = gst_element_factory_make ("videoconvert", nullptr);
  GstElement *time = gst_element_factory_make ("timeoverlay", nullptr);
  GstElement *sink = gst_element_factory_make ("glimagesink", nullptr);
  m_video_sinks.insert(0, sink);

  // qDebug () << "[setGstFileVideo] rtspUrl = " << rtspUrl;
  g_object_set (src, "location", "C:\\test-vids\\sintel-trailer-720p.mp4", nullptr);
  g_object_set (time, "font-desc", "Sans, 24", nullptr);
  //g_object_set (vqueue, "ring-buffer-max-size", (guint64)40000000, nullptr);
  g_object_set (vqueue,
                "max-size-bytes", (guint64)128 * 1024 * 1024, // 128 MB buffer
                //"ring-buffer-max-size", (guint64)128 * 1024, // 128 KB buffer
                "use-buffering", true,
                nullptr);

  gst_bin_add_many (GST_BIN (pipeline), src, demux, vdec, vqueue, vconv, time, sink, nullptr);

  gst_element_link_many (src, demux, nullptr);
  // link demux to vdec in a cb, when pad is created
  g_signal_connect (demux, "pad-added", G_CALLBACK (pad_added_cb), vdec);
  // link rest of the pipeline
  gst_element_link_many (vdec, vqueue, vconv, time, sink, nullptr);

  WId xwinid = m_videoDisplays[0]->winId();
  gst_video_overlay_set_window_handle (GST_VIDEO_OVERLAY (sink), xwinid);

  gst_element_set_state (pipeline, GST_STATE_PAUSED);
  qDebug() << "[setGstFileVideo] All done, pipeline is in PAUSED state";

  return;
}


void VideoPlayer::setGstRtspVideo ()
{
    setGstRtspVideo(0);
}

void VideoPlayer::setGstRtspVideo (int video_index)
{
  GstElement *src = gst_element_factory_make ("rtspsrc", nullptr);
  GstElement *vdepay = gst_element_factory_make ("rtph264depay", nullptr);
  GstElement *vdec = gst_element_factory_make ("avdec_h264", nullptr);
  GstElement *vqueue = gst_element_factory_make ("queue2", nullptr);
  GstElement *vconv = gst_element_factory_make ("videoconvert", nullptr);
  GstElement *time = gst_element_factory_make ("timeoverlay", nullptr);
#ifdef _WIN32
  GstElement *sink = gst_element_factory_make ("d3dvideosink", nullptr);
#else
  GstElement *sink = gst_element_factory_make ("xvimagesink", nullptr);
#endif
  m_video_sinks.insert(video_index, sink);

  //g_object_set (src, "location", "rtsp://184.72.239.149/vod/mp4:BigBuckBunny_115k.mov", nullptr);
  QString rtspUrl = m_rtspUrls[video_index];
  qDebug () << "[setGstRtspVideo] rtspUrl = " << rtspUrl;
  g_object_set (src, "location", rtspUrl.toStdString().c_str(), nullptr);
  g_object_set (src, "latency", 300, nullptr);
  g_object_set (time, "font-desc", "Sans, 24", nullptr);
  g_object_set (vqueue, "max-size-bytes", (guint64)40000000, nullptr);

  if (!rate_control) {
    g_object_set (src, "onvif-rate-control", FALSE, nullptr);
  }

  if (use_tcp) {
    g_object_set (src, "protocols", GST_RTSP_LOWER_TRANS_TCP, nullptr);
  }

  gst_bin_add_many (GST_BIN (pipeline), src, vdepay, vdec, vqueue, vconv, time, sink, nullptr);
  // link src to vdepay in a cb, when pad is created
  g_signal_connect (src, "pad-added", G_CALLBACK (pad_added_cb), vdepay);
  // link rest of the pipeline
  gst_element_link_many (vdepay, vdec, vqueue, vconv, time, sink, nullptr);

  WId xwinid = m_videoDisplays[video_index]->winId();
  gst_video_overlay_set_window_handle (GST_VIDEO_OVERLAY (sink), xwinid);

  return;
}


void VideoPlayer::send_playback_speed_seek_event () {
  return send_playback_speed_seek_event(0);
}

void VideoPlayer::send_playback_speed_seek_event (gint64 offset) {
    bool absolute = false;
    return send_playback_speed_seek_event(offset, absolute);
}

void VideoPlayer::send_playback_speed_seek_event_absolute(gint64 position) {
    bool absolute = true;
    return send_playback_speed_seek_event(position, absolute);
}

void VideoPlayer::send_playback_speed_seek_event (gint64 offset, bool absolute) {
  // offset is (eventually) an seek offset, eg seek 10 seconds later
  gint64 position = 0;
  GstEvent *seek_event;
  GstSeekFlags seek_flags = (GstSeekFlags) (frame_seek_flag| GST_SEEK_FLAG_FLUSH | GST_SEEK_FLAG_ACCURATE);
  GstSeekType seek_type = GST_SEEK_TYPE_SET;

  if (!pipeline) {
      return;
  }

  if (offset == 0) {
    seek_type = GST_SEEK_TYPE_NONE;
  }

  qDebug() << "[send_playback_speed_seek_event] Entering function";

  /* Obtain the current position, needed for the seek event */
  if (!absolute) {
      if (!gst_element_query_position (pipeline, GST_FORMAT_TIME, &position)) {
        g_printerr ("Unable to retrieve current positio, using 0 for seek_start_position.\n");
        // return;
      }
  }

  // apply position offset
  position = position + offset;
  if (position < 0) {
    position = 0;
  }

  double rate = playSpeed;
  if (reversePlayback) {
    rate = -1.0 * rate;
  }
  /* Create the seek event */
  if (rate > 0) {
    // playing forward
    qDebug () << "Playing forward: speed=" << rate;
    seek_event = gst_event_new_seek (rate,
                                     GST_FORMAT_TIME,
                                     seek_flags,
                                     seek_type, position,
                                     GST_SEEK_TYPE_NONE, 0);
  } else {
    // playing backwards
    qDebug ( )<< "Playing backwards: speed=" << rate;
    seek_event = gst_event_new_seek (rate,
                                     GST_FORMAT_TIME,
                                     seek_flags,
                                     seek_type, 0,
                                     seek_type, position);
  }


  /* obtain the sink through which we will send the seek events */
  // g_object_get (pipeline, "my-video-sink", video_sink, nullptr);

  /* Send the event */
  gst_element_send_event (pipeline, seek_event);

  qDebug() << "[send_playback_speed_seek_event] Done.";
  return;
}

void VideoPlayer::play()
{
  GstStateChangeReturn sret;
  bool firstRun = false;
  if (!pipeline) {
    firstRun = true;
    initGst();
    for (int i = 0; i < m_rtspUrls.length(); i++) {
        setGstRtspVideo(i);
    }
    //setGstFileVideo();
    //setGstTestVideo();

    sret = gst_element_set_state (pipeline, GST_STATE_PAUSED);
    // wait until the pipeline is really in PAUSED state
    sret = gst_element_get_state(pipeline, nullptr, nullptr, 1 * GST_SECOND);
    print_status_of_all();
  }

  // run the pipeline
  sret = gst_element_set_state (pipeline, GST_STATE_PLAYING);
  if (sret == GST_STATE_CHANGE_FAILURE) {
    gst_element_set_state (pipeline, GST_STATE_NULL);
    gst_object_unref (pipeline);
    // Exit application
    qCritical("Could not set pipeline to PLAYING state");
    //QTimer::singleShot(0, QApplication::activeWindow(), SLOT(quit()));
  }

  // when instanciating the pipeline, first set some options through a seek event
  // be sure to wait that the pipeline is playing (above set_state is async)
  if (firstRun) {
    sret = gst_element_get_state(pipeline, nullptr, nullptr, 10 * GST_SECOND);
    send_playback_speed_seek_event();
  }

  m_playButton->setIcon(style()->standardIcon(QStyle::SP_MediaPause));
  m_stopButton->setEnabled(true);
  m_seekBackwardButton->setEnabled(true);
  m_seekForwardButton->setEnabled(true);
  m_nextFrameButton->setEnabled(true);
  for (int i = 0; i < m_urlEdits.length(); i++) {
      m_urlEdits[i]->setEnabled(false);
  }
  m_addVideoButton->setEnabled(false);
  playing = true;
  return;
}

void VideoPlayer::pause()
{
  qDebug() << "PAUSE called";
  //GST_DEBUG_BIN_TO_DOT_FILE((GstBin *)pipeline, GST_DEBUG_GRAPH_SHOW_ALL, "pipeline");
  GstStateChangeReturn sret = gst_element_set_state (pipeline, GST_STATE_PAUSED);
  if (sret == GST_STATE_CHANGE_FAILURE) {
    gst_element_set_state (pipeline, GST_STATE_NULL);
    gst_object_unref (pipeline);
    // Exit application
    QTimer::singleShot(0, QApplication::activeWindow(), SLOT(quit()));
  }
  playing = false;
  m_playButton->setIcon(style()->standardIcon(QStyle::SP_MediaPlay));
  return;
}

void VideoPlayer::playPause()
{
  if (playing) {
    qDebug() << "toggle: play --> pause";
    VideoPlayer::pause();
  } else {
    qDebug() << "toggle: pause --> play";
    VideoPlayer::play();
  }
  return;
}

void VideoPlayer::stop()
{
  qDebug() << "[STOP function] called";

  if (pipeline) {
    qDebug() << "[STOP function] detected a running pipeline";
    gst_element_set_state (pipeline, GST_STATE_NULL);
    gst_object_unref (pipeline);
    pipeline = nullptr;
    qDebug() << "[STOP function] stopped and dropped pipeline";
  } else {
    qDebug() << "[STOP function] no running pipeline";
  }

  // Reinit interface
  playing = false;
  m_playButton->setIcon(style()->standardIcon(QStyle::SP_MediaPlay));
  for (int i = 0; i < m_videoDisplays.length(); i++) {
      m_videoDisplays[i]->repaint();
  }
  m_stopButton->setEnabled(false);
  m_seekBackwardButton->setEnabled(false);
  m_seekForwardButton->setEnabled(false);
  m_nextFrameButton->setEnabled(false);
  for (int i = 0; i < m_urlEdits.length(); i++) {
      m_urlEdits[i]->setEnabled(true);
  }
  m_addVideoButton->setEnabled(true);

  return;
}



void VideoPlayer::debugSlot()
{
  qDebug() << "DEBUG";
  print_status_of_all();
  qDebug() << "Rate-Control: " << rate_control;
  qDebug() << "Use TCP? " << use_tcp;
}

void VideoPlayer::disableRateControl(bool is_disabled)
{
  rate_control = !is_disabled;
}

void VideoPlayer::useTcp(bool value)
{
  use_tcp = value;
}

void VideoPlayer::setFrameMode(QString frameMode)
{
  qDebug() << "setFrameMode: " << frameMode << " selected";
  if (frameMode == "All") {
    frame_seek_flag = GST_SEEK_FLAG_TRICKMODE;
  } else if (frameMode == "Intra") {
    frame_seek_flag = GST_SEEK_FLAG_TRICKMODE_KEY_UNITS;
  } else if (frameMode == "Predicted") {
    frame_seek_flag = GST_SEEK_FLAG_TRICKMODE_PREDICTED;
  } else {
    qDebug() << "setFrameMode - Don't know what was clicked";
    frame_seek_flag = GST_SEEK_FLAG_NONE;
  }

  if (!reversePlayback) {
    double rate = playSpeed;
    GstEvent *seek_event = gst_event_new_seek (rate, GST_FORMAT_TIME,
                                               (GstSeekFlags) (GST_SEEK_FLAG_FLUSH | frame_seek_flag),
                                               GST_SEEK_TYPE_NONE, 0, GST_SEEK_TYPE_NONE, 0);
    gst_element_send_event (pipeline, seek_event);
  } else {
    qDebug () << "TODO: handle frame mode when playing backwards";
  }
  return;
}

void VideoPlayer::setRtspUrl(QString url)
{
  QObject *obj = sender();
  int video_index = -1;
  video_index = m_urlEdits.indexOf((QLineEdit *)obj);
  m_rtspUrls[video_index] = url;
  qDebug() << "setRtspUrl: video #" << video_index << "url set to " <<url;
}

void VideoPlayer::setPlaybackSpeed(double speed)
{
  playSpeed = speed;
  qDebug() << "[setPlaybackSpeed] speed = " << speed;
  send_playback_speed_seek_event();
}

void VideoPlayer::setReversePlayback(bool isChecked)
{
  reversePlayback = isChecked;
  qDebug() << "[setReversePlayback] reversePlayback = " << reversePlayback;
  send_playback_speed_seek_event();
}

void VideoPlayer::seekBackward10()
{
  qDebug() << "[seekBackward10] seek called";
  send_playback_speed_seek_event(-10 * GST_SECOND);
  qDebug() << "[seekBackward10] seek done.";

  return;
}


void VideoPlayer::seekForward10()
{
  qDebug() << "[seekForward10] seek called";
  send_playback_speed_seek_event(10 * GST_SECOND);
  qDebug() << "[seekBackward10] seek done";

  return;
}

void VideoPlayer::updatePosition()
{
    gint64 current_position = -1;
    QString pos_text;

    if (playing) {
        if (gst_element_query_position (pipeline, GST_FORMAT_TIME, &current_position)) {
            // position returned by xebra streams is a ntp timestamp, in nanoseconds
            // NTP epoch is 1/1/1900, unix epoch is 1/1/1970
            // we convert it to a unix timestamp by substracting 70 years
            gint64 milliseconds = current_position / 1000000;
            gint64 unix_milliseconds = milliseconds - 2208988800000;
            QDateTime datetime = QDateTime::fromMSecsSinceEpoch(unix_milliseconds).toUTC();
            pos_text = "Position: " + QString::number(current_position);
            m_positionLabel->setText(pos_text);
            if (!m_position->hasFocus()) {
                m_position->setDateTime(datetime);
            }
        }
    }
}

void VideoPlayer::seekToPosition()
{
    qDebug() << "w00t, position edited to" << m_position->dateTime();

    QDateTime datetime = m_position->dateTime();
    // convert datetime to a suitable gstreamer position
    // eg. nanoseconds since ntp epoch
    gint64 unix_msecs = datetime.toMSecsSinceEpoch();
    gint64 ntp_msecs = unix_msecs + 2208988800000;
    gint64 position = ntp_msecs * 1000000;
    qDebug() << "      seeking to position " << position;

    send_playback_speed_seek_event_absolute(position);
}

void VideoPlayer::nextFrame()
{
    qDebug() << "Next frame (forward)";
    if (!pipeline) {
        qDebug() << "    Gstreamer pipeline not running, aborting.";
        return;
    }
    if (m_video_sinks.empty()) {
        qDebug () << "    No video sink found, aborting.";
    }
    pause();
    GstEvent *step_event;
    //step_event = gst_event_new_step (GST_FORMAT_BUFFERS, 1, ABS (data->rate), true, false);
    step_event = gst_event_new_step (GST_FORMAT_BUFFERS,
                                     1,      // nb of steps
                                     1.0,    // rate
                                     true,   // flush
                                     false); //
    gst_element_send_event (pipeline, step_event);


}

void VideoPlayer::addRtspUrlEdit()
{
    qDebug () << "Adding a LineEdit widget for a new rtsp video";
    int nb_video_max = 6;
    if (m_urlEdits.length() < nb_video_max) {
        QString default_rtsp_url = "";
        QLineEdit *qle = new QLineEdit(this);
        // fix tab focus order
        QWidget::setTabOrder(m_urlEdits.last(), qle);
        QWidget::setTabOrder(qle, m_addVideoButton);
        m_urlEdits.append(qle);
        m_rtspUrls.append(default_rtsp_url);
        connect (qle, SIGNAL(textEdited(QString)), this, SLOT(setRtspUrl(QString)));
        qle->setText(default_rtsp_url);
        m_urlsLayout->addWidget(qle);
        qle->setFocus();

        // Create new widget to hold the video
        QWidget *w = new QWidget();
        w->setMinimumHeight(240);
        m_videoDisplays.append(w);
        int n_videos = m_playersLayout->count();
        int row = n_videos / 3;
        int col = n_videos % 3;
        m_playersLayout->addWidget(w, row, col);

    } else {
        QMessageBox msgBox;
        QString msg = "Cannot use more than " + QString::number(nb_video_max) + " video streams.";
        msgBox.setText(msg);
        msgBox.exec();
        return;
    }
}
